/*= INTERFACE 9714 ========================================================*/
/* LabWindows Instrument Driver                                            */
/* Original Release: V1.0  06.12.95                                        */
/* By: M.Westermann                                                        */
/* Originally written in C                                                 */
/* Modification History: S.Thiess V1.1  25.01.1999                         */
/*=========================================================================*/

/*---- LabWindows CVI Header----*/
#include <utility.h>
#include <rs232.h>
#include <formatio.h>
#include <ansi_c.h>

/*---- LabWindows DOS Header----
#include "C:\LW\INCLUDE\formatio.h"
#include "C:\LW\INCLUDE\lwsystem.h"
#include "C:\LW\INCLUDE\rs232.h"*/

#include "bp9714.h"


/*= STATIC VARIABLES ======================================================*/
/* for CVI Only	   */
static int escape_on_bit;
static int escape_off_bit;
static double escape_port_delay;
static int ComType;							/* RS232, RS48X */

static int setup_first ;                  	/* remark the first initial*/
static double std_timeout;                	/* remark the user timeout*/
static int max_bcc_repeat;                	/* Max-Bcc-Repeat */
static int bcc_repeat;                    	/* Bcc-Repeat counter*/
static int bcc_state;                     	/* Blockcheck state ON or OFF*/
static int instr_cnt;
static int bp9714_err;
static int init_g_adr,init_u_adr,init_all_counts;
int init_all_flag;

static int address[bp9714_MAX_INSTR + 1];  	 /*Address*/
static char cmd[bp9714_MAX_CMD+1];
static char buf[bp9714_MAX_CMD+1];
static char out_buf[bp9714_MAX_CMD+1];       /* buffer for serial send */
static char in_buf[bp9714_MAX_CMD+1];        /* buffer for serial receive */
static char swap_buf[bp9714_MAX_CMD+1];      /* global serial buffer */
static int port_addr[4] ;                    /* RS232-Port Array*/
static int int_level[4] ;                    /* RS232-Interrupt-Level Array*/
static int dev_port[bp9714_MAX_INSTR];       /* Device Port*/
static int dev_group_addr[bp9714_MAX_INSTR]; /* Device Group serial Adress*/
static int dev_user_addr[bp9714_MAX_INSTR];  /* Device User serial Adress */


static char * func_system_keyboard[2];
static char * func_cal_input_meas[2];
static char * func_cal_unit[2];
static char * func_aver_ctrl[2];
static char * func_shape[5];
static char * func_output_type[2];
static char * func_output_level[6];
static char * func_analog_type[2];
static char * func_input_ref_point[2];
static char * func_input_range[17];
static char * func_input_filter[2];
static char * func_comp_type[2];
static char * func_comp_ctrl[4];



/*= UTILITY ROUTINES ======================================================*/

void bp9714_setup_arrays (void);
int bp9714_invalid_integer_range (int, int, int, int);
int bp9714_invalid_long_range (long, long, long, int);
int bp9714_invalid_real_range (double, double, double, int);
int bp9714_get_eoc(int );
int bp9714_get_tare_state(int ); 
int bp9714_send_x3_28(int,char*,char*);
int bp9714_send_x3_28_message(int,char*,char*);
int bp9714_receive_x3_28(int,char*,char*,char*);
int bp9714_receive_and_ack(int,char*);
int bp9714_close_instr (int);
int bp9714_device_closed (int);
int bp9714_open_instr(int,long,int,int);
int com_wrt_byte(int port,int send_byte);
int com_wrt(int port,char* send_string);
unsigned char bp9714_calculate_bcc(char* );


/*=========================================================================*/
/* This function opens a com port for the instrument module                */
/*=========================================================================*/
int _VI_FUNC bp9714_init_device(int port,int baud_rate,double timeout,int set_bcc_state,
                   int g_addr,int u_addr,char* model,int reset_flag,
                   int *instrID)
{
    int ID;

    bp9714_err = 0;
    bp9714_setup_arrays ();

    if (bp9714_invalid_integer_range (port, 1,4, -1) != 0)
     return bp9714_err;
    if (bp9714_invalid_long_range (baud_rate, 1200, 38400, -2) != 0)
     return bp9714_err;
    if (bp9714_invalid_real_range (timeout, 0.0, 72000.0, -3) != 0)
     return bp9714_err;
    if (bp9714_invalid_integer_range (set_bcc_state, 0,1, -4) != 0)
     return bp9714_err;
    if (bp9714_invalid_integer_range (g_addr, 0,15, -5) != 0)
     return bp9714_err;
    if (bp9714_invalid_integer_range (u_addr, 0,15, -6) != 0)
     return bp9714_err;
    if (bp9714_invalid_integer_range (reset_flag, 0,1, -7) != 0)
     return bp9714_err;

    ID = bp9714_open_instr(port,baud_rate,g_addr,u_addr);
    if(ID <= 0)
     return bp9714_err;

    SetComTime (port, timeout);
    std_timeout = timeout;
    escape_port_delay = (double) (1.0/baud_rate) * 10.0;

	ComType = bp9714_RS232;					
    if (rs232err != 0)
        return (rs232err);

    max_bcc_repeat = 5;       				/* default*/
    bcc_state = set_bcc_state;

    if (bp9714_receive_scpi(ID,"*IDN?",model) != 0)
    {
     bp9714_close_instr(ID);
     return bp9714_err;
    }
    if (reset_flag == 1)
    {
     if(bp9714_send_scpi(ID,"*RST") != 0) 	/* Reset*/
     {
      bp9714_close_instr(ID);
      return bp9714_err;
     }
    }
    *instrID = ID;
    return(0);
}


/*=========================================================================*/
/* This function opens a com port for the instrument module                */
/*=========================================================================*/
int _VI_FUNC bp9714_init_all_devices(int port,int baud_rate,double timeout,int set_bcc_state,
                   int instrID[],int instr_g_adr[],int instr_u_adr[],
                   int *instr_numbers)
{
    int ID;

    bp9714_err = 0;

    if (bp9714_invalid_integer_range (port, 1,4, -1) != 0)
     return bp9714_err;
    if (bp9714_invalid_long_range (baud_rate, 1200, 38400, -2) != 0)
     return bp9714_err;
    if (bp9714_invalid_real_range (timeout, 0.0F, 72000.0F, -3) != 0)
     return bp9714_err;
    if (bp9714_invalid_integer_range (set_bcc_state, 0,1, -4) != 0)
     return bp9714_err;
     
	escape_port_delay = (double) (1.0/baud_rate) *10.0;
	
    switch(init_all_flag)
    {
     case 0:
      bp9714_setup_arrays ();   		/* Reset the Arrays*/
      max_bcc_repeat = 5;       		/* default*/
      bcc_state = set_bcc_state;

      init_g_adr = 0;           		/* Start Adress */
      init_u_adr = 0;
      init_all_counts = 0;      		/* Counter Reset*/
      init_all_flag = 1;        		/* Set Init-Flag next Step*/
      break;
     case 1:
      init_u_adr++;               		/* Next User Address*/
      if(init_u_adr == 16)
      {
       init_all_flag = 3;         		/* Set Init-Flag next Step*/
      }
      break;
     case 2:
      init_g_adr++;               		/* Next Group Address*/
      if(init_g_adr == 15)
       init_all_flag = 3;         		/* Set Init-Flag next Step*/
      else
      {
       init_u_adr = 0;            		/* Reset User Address*/
       init_all_flag = 1;         		/* Set Init-Flag next Step*/
      }
      break;
     case 3:                       		/* All Address search*/
      init_all_flag = 0;           		/* Reset the Control Flag */
      SetComTime (port, timeout);  		/* reset the default Timeout Value*/
      std_timeout = timeout;       		/* store teh timeout Value*/
      
      if (rs232err != 0)
        return (rs232err);
      *instr_numbers = init_all_counts; /* restore the number of
                                           found instruments*/
      return (0);
//      break;
    }
    ID = bp9714_open_instr(port,baud_rate,init_g_adr,init_u_adr);
     if (rs232err != 0)
       return (rs232err);

    SetComTime (port, (timeout/10.0F)+0.1F);/* smal timeout => fast search*/

    if (bp9714_send_scpi(ID,"*CLS") != 0) 	/* Send Message */
    {
     bp9714_close_instr(ID);             	/* Address not found*/
    }
    else
    {
     instrID[init_all_counts] = ID;   		/* Address found*/
     instr_g_adr[init_all_counts] = init_g_adr;
     instr_u_adr[init_all_counts] = init_u_adr;
     init_all_counts++;
    }
    return(360);
}


/* ========================================================================= */
/*  This function switches the Escape Bits for RS485 Halfduplex Mode         */
/* ========================================================================= */
int _VI_FUNC bp9714_init_rs485(int ctrl_bit,int ctrl_state)
{
    if (bp9714_invalid_integer_range (ctrl_bit, 0,1, -1) != 0)
     return bp9714_err;
    if (bp9714_invalid_integer_range (ctrl_state, 0,1, -2) != 0)
     return bp9714_err;

	ComType = bp9714_RS48X;							

    /*   for CVI only*/

    if(ctrl_bit == bp9714_RS485_CTRL_RTS)  			/*   RTS Bit*/
    {
     if(ctrl_state == bp9714_RS485_CTRL_LOWACTIVE) 	/*  Low Active*/
     {
      escape_on_bit = CLRRTS;
      escape_off_bit = SETRTS;
     } 
     else
     {
      escape_on_bit = SETRTS;
      escape_off_bit = CLRRTS;
     } 
    }
    else    					           			/*   DTR BIT*/
    {
     if(ctrl_state == bp9714_RS485_CTRL_LOWACTIVE)  /*  Low Active*/
     {
      escape_on_bit = CLRDTR;
      escape_off_bit = SETDTR;
     } 
     else
     {
      escape_on_bit = SETDTR;
      escape_off_bit = CLRDTR;
     } 
    }
    return 0;
}


/* ========================================================================= */
/*  This function switches the calibration state ON/OFF                      */
/* ========================================================================= */
int _VI_FUNC bp9714_cal_state(int instrID,int state)
{
    bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;
    if (bp9714_invalid_integer_range (state, 0, 1,  -2) != 0)
     return bp9714_err;

    Fmt (cmd, "%s<CAL:STAT %i",state);
    if (bp9714_send_scpi (instrID, cmd) != 0)
     return bp9714_err;

    return (0);
}


/* ========================================================================= */
/*  This function reads the calibration state                                */
/* ========================================================================= */
int _VI_FUNC bp9714_read_cal_state(int instrID,int *state)
{
 
    bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;
   
    if(bp9714_receive_scpi(instrID,"CAL:STAT?",buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%d",state); 

    return (0);
}


/* ========================================================================= */
/*  This function writes the calibration values                              */
/* ========================================================================= */
int _VI_FUNC bp9714_cal_input(int instrID,double low_cal_value,double high_cal_value,
                                 double low_scale_value,double high_scale_value)
{
    bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;

    /* Calibration Values low and high*/
    
    Fmt (cmd, "%s<CAL:DATA:SENS:LOW %f",low_cal_value);
    if (bp9714_send_scpi (instrID, cmd) != 0)
     return bp9714_err;

    Fmt (cmd, "%s<CAL:DATA:SENS:HIGH %f",high_cal_value);
    if (bp9714_send_scpi (instrID, cmd) != 0)
     return bp9714_err;

    /* Scale Values low and high*/

    Fmt (cmd, "%s<CAL:DATA:SCAL:LOW %f",low_scale_value);
    if (bp9714_send_scpi (instrID, cmd) != 0)
     return bp9714_err;

    Fmt (cmd, "%s<CAL:DATA:SCAL:HIGH %f",high_scale_value);
    if (bp9714_send_scpi (instrID, cmd) != 0)
     return bp9714_err;

    return (0);
}


/* ========================================================================= */
/*  This function reads the calibration values                               */
/* ========================================================================= */
int _VI_FUNC bp9714_read_cal_input(int instrID,double *low_cal_value,double *high_cal_value,
                          double *low_scale_value,double *high_scale_value)
{
    bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;

    /* Calibration value low */
  
	if(bp9714_receive_scpi(instrID,"CAL:DATA:SENS:LOW?",buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%f",low_cal_value);     
    
	/* Calibration value high */

    if(bp9714_receive_scpi(instrID,"CAL:DATA:SENSE:HIGH?",buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%f",high_cal_value); 

    /* Scale value low */

	if(bp9714_receive_scpi(instrID,"CAL:DATA:SCAL:LOW?",buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%f",low_scale_value); 
    
	/* Scale value high */

	if(bp9714_receive_scpi(instrID,"CAL:DATA:SCAL:HIGH?",buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%f",high_scale_value); 

    return (0);
}


/* ========================================================================= */
/*  This function inputs the calibration values of the analogue Output       */
/* ========================================================================= */
int _VI_FUNC bp9714_cal_analogue_output(int instrID,double low_value,double low_scale,
							   double high_value, double high_scale)
{
    bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;

    /* Calibration Values low and high for Analogue Output*/
    
    Fmt (cmd, "%s<CAL:DATA:SOUR:LOW %f%s%f",low_scale,",",low_value);
    if (bp9714_send_scpi (instrID, cmd) != 0)
     return bp9714_err;

    Fmt (cmd, "%s<CAL:DATA:SOUR:HIGH %f%s%f",high_scale,",",high_value);
    
    if (bp9714_send_scpi (instrID, cmd) != 0)
     return bp9714_err;

    return (0);
}


/* ========================================================================= */
/*  This function reads the calibration values of the analogue Output        */
/* ========================================================================= */
int _VI_FUNC bp9714_read_cal_analogue_output(int instrID,double *low_value, double *low_scale,
									double *high_value, double *high_scale)
{
    bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;

    /* Calibration Values low and high for Analogue Output*/
	
	if(bp9714_receive_scpi(instrID,"CAL:DATA:SOUR:LOW?",buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%f,%f",low_scale,low_value);     
    
	if(bp9714_receive_scpi(instrID,"CAL:DATA:SOUR:HIGH?",buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%f,%f",high_scale,high_value);     
    
    return (0);
}


/* ========================================================================= */
/*  This function measures the calibration values                            */
/* ========================================================================= */
int _VI_FUNC bp9714_cal_input_meas(int instrID,int meas_mode)
{
    bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;

    if (bp9714_invalid_integer_range (meas_mode,0, 1,  -2) != 0)
     return bp9714_err;

    if (bp9714_send_scpi (instrID, func_cal_input_meas[meas_mode]) != 0)
     return bp9714_err;

    return (0);
}


/* ========================================================================= */
/*  This function reads the calibration values                               */
/* ========================================================================= */
int _VI_FUNC bp9714_read_cal_input_meas(int instrID,double *kal1, double *kal2)
{
    bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;

	if(bp9714_receive_scpi(instrID,"CAL:DATA:SENSE:LOW?",buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%f",kal1);     
          
	if(bp9714_receive_scpi(instrID,"CAL:DATA:SENS:HIGH?",buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%f",kal2);     
          
    return (0);
}


/* ========================================================================= */
/*  This function sets the calibration unit                                  */
/* ========================================================================= */
int _VI_FUNC bp9714_cal_set_unit(int instrID,int cunit)
{
    bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;

    if (bp9714_invalid_integer_range (cunit,0, 1,  -2) != 0)
     return bp9714_err;

    if (bp9714_send_scpi (instrID, func_cal_unit[cunit]) != 0)
     return bp9714_err;

    return (0);
}


/* ========================================================================= */
/*  This function reads the calibration unit                                 */
/* ========================================================================= */
int _VI_FUNC bp9714_read_cal_unit(int instrID,int *cunit)
{
    bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;

	if(bp9714_receive_scpi(instrID,"CAL:MATH:EXPR?",buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%d",cunit);     
	
    return (0);
}


/* ========================================================================= */
/*  This function accepts the  calibration values                            */
/* ========================================================================= */
int _VI_FUNC bp9714_cal_accept_datas(int instrID)
{
    char cbuf[10];
    int flag;

    bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;

    Fmt (cmd, "%s<CAL:ALL?");
    if(bp9714_receive_scpi(instrID,"CAL:ALL?",cbuf) != 0)/* read flag */
     return bp9714_err;

    Scan(cbuf,"%s>%i",&flag);
    if(flag == 1)    									/* No Correct Calibration values*/
     return (330);

    return (0);
}


/* ========================================================================= */
/*  This function configures the device                                      */
/* ========================================================================= */
int _VI_FUNC bp9714_config(int instrID,int aver_ctrl,int aver_counts,int meas_mode,
                  int display_format,int device_unit, int start_mode)
{
    bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;
    if (bp9714_invalid_integer_range (aver_ctrl,0 , 1 , -2) != 0)
     return bp9714_err;
    if (bp9714_invalid_integer_range (aver_counts,1 , 9999 , -3) != 0)
     return bp9714_err;
    if (bp9714_invalid_integer_range (meas_mode,0 ,1 , -4) != 0)
     return bp9714_err;
    if (bp9714_invalid_integer_range (display_format,0 ,6 , -5) != 0)
     return bp9714_err;
    if (bp9714_invalid_integer_range (device_unit,0 ,43 , -5) != 0)
     return bp9714_err;
    if (bp9714_invalid_integer_range (start_mode,0 ,1 , -6) != 0)
     return bp9714_err;
     
    Fmt (cmd, "%s<SENS:AVER:COUN %i",aver_counts);
    if (bp9714_send_scpi (instrID, cmd) != 0)
     return bp9714_err;

    if (bp9714_send_scpi (instrID, func_aver_ctrl[aver_ctrl]) != 0)
     return bp9714_err;

    Fmt (cmd, "%s<INIT:CONT %i",meas_mode);
    if (bp9714_send_scpi (instrID, cmd) != 0)
     return bp9714_err;

    Fmt (cmd, "%s<FORM ASC,%i",display_format);
    if (bp9714_send_scpi (instrID, cmd) != 0)
     return bp9714_err;

    Fmt (cmd, "%s<CALC:MATH:EXPR %i",device_unit);
    if (bp9714_send_scpi (instrID, cmd) != 0)
     return bp9714_err;
     
    Fmt (cmd, "%s<SENS:START:STATE %i",start_mode);
    if (bp9714_send_scpi (instrID, cmd) != 0)
     return bp9714_err;
    
    return (0);
}

/* ========================================================================= */
/*  This function reads the configuration of the device                      */
/* ========================================================================= */
int _VI_FUNC bp9714_read_config(int instrID,int *aver_ctrl,int *aver_counts,int *meas_mode,
                       int *display_format,int *device_unit, int *start_mode)
{
    char hilf[10];
    bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;
	
	if(bp9714_receive_scpi(instrID,"SENS:AVER:COUNT?",buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%d",aver_counts);     
          
	if(bp9714_receive_scpi(instrID,"SENS:AVER:TCON?",buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%s",hilf);     
	
	if(strcmp(hilf,"MOV")==0)   /* Decoding the averaging type */
		*aver_ctrl = 0;
	else
		*aver_ctrl = 1;
    
	/* Reading measurement mode */

	if(bp9714_receive_scpi(instrID,"INIT:CONT?",buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%d",meas_mode);     
  
	/* Reading display format */

	if(bp9714_receive_scpi(instrID,"FORM?",buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>ASC,%d",display_format);     
       
	/* Reading measurement unit */

	if(bp9714_receive_scpi(instrID,"CALC:MATH:EXPR?",buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%d",device_unit);     
       
	/* Reading the start mode*/

	if(bp9714_receive_scpi(instrID,"SENS:START:STATE?",buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%d",start_mode); 
    return (0);
}	
	

/* ========================================================================= */
/*  This function writes the configuration of the output                     */
/* ========================================================================= */
int _VI_FUNC bp9714_config_output(int instrID,int shape,int supply)
{
    bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;
    
    
    if (bp9714_send_scpi (instrID, func_shape[shape]) != 0)
     return bp9714_err;

    if ( (supply >= 0) && (supply <= 2))
    {
     if (bp9714_send_scpi (instrID, func_output_type[0]) != 0) /*Asym.*/
      return bp9714_err;
    }
    else
    {
     if (bp9714_send_scpi (instrID, func_output_type[1]) != 0) /*Sym.*/
      return bp9714_err;
    }
    if (bp9714_send_scpi (instrID, func_output_level[supply]) != 0)
     return bp9714_err;

    return (0);
}


/* ========================================================================= */
/*  This function reads the configuration of the output                      */
/* ========================================================================= */
int _VI_FUNC bp9714_read_config_output(int instrID,int *shape,int *supply)
{
    char hilf[20];
    bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;
    

	if(bp9714_receive_scpi(instrID,"OUTP:TYPE?",buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%s",hilf);     
           
	if(strcmp(hilf,"UNB")==0)	  /* Decode excitation type */
		*shape = 0;				  /* unbalanced */
	else 
		*shape = 1;				  /* balanced */

	if(bp9714_receive_scpi(instrID,"SOUR:VOLT?",buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%s",hilf);  

	/* Decode exitation voltage */
	
	if((strcmp(hilf,"2.5V")==0)&&(*shape == 0))
		*supply = 0;
	else
	if((strcmp(hilf,"5V")==0)&&(*shape == 0))
		*supply = 1;
	else
	if((strcmp(hilf,"10V")==0)&&(*shape == 0))
		*supply = 2;
	else
	if((strcmp(hilf,"5V")==0)&&(*shape == 1))
		*supply = 3;
	else
	if((strcmp(hilf,"10V")==0)&&(*shape == 1))
		*supply = 4;
	else
	if((strcmp(hilf,"20V")==0)&& (*shape == 1))
		*supply = 5;
	else
		*supply = 0;
    

	if(bp9714_receive_scpi(instrID,"SOUR:FUNC?",buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%s",hilf);     
  	if((strcmp(hilf,"DC")==0))
		*shape = 0;

  	if((strcmp(hilf,"LOW")==0))
		*shape = 1;

  	if((strcmp(hilf,"DCR")==0))
		*shape = 2;

  	if((strcmp(hilf,"ACR")==0))
		*shape = 3;

  	if((strcmp(hilf,"ALTR")==0))
		*shape = 4;


    return (0);
}


/* ========================================================================= */
/*  This function configures the analog output function                      */
/* ========================================================================= */
int _VI_FUNC bp9714_config_analog_output(int instrID,int analog_type)
{
    bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;
    if (bp9714_invalid_integer_range (analog_type,0 , 1 , -2) != 0)
     return bp9714_err;
    if (bp9714_send_scpi (instrID, func_analog_type[analog_type]) != 0)
     return bp9714_err;

    return (0);
}


/* ========================================================================= */
/*  This function reads the configuration of the analogue output             */
/* ========================================================================= */
int _VI_FUNC bp9714_read_config_analog_output(int instrID,int *analog_type)
{
    char hilf[10];
    bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;
	if(bp9714_receive_scpi(instrID,"SOUR:FUNC:MODE?",buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%s",hilf);     
           
	/* decode output type */
	
	if(strcmp(hilf,"CURR")==0)
		*analog_type = 0;		/* current */
	else 
		*analog_type = 1;		/* voltage */


    return (0);
}


/* ========================================================================= */
/*  This function configures the input function of the device                */
/* ========================================================================= */
int _VI_FUNC bp9714_config_input(int instrID,int ref_point,int filter,int range)
{
    bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;
    if (bp9714_invalid_integer_range (ref_point,0 , 1 , -2) != 0)
     return bp9714_err;
    if (bp9714_invalid_integer_range (filter,0 , 1 , -3) != 0)
     return bp9714_err;
    if (bp9714_invalid_integer_range (range,0 , 16 , -4) != 0)
     return bp9714_err;

    if (bp9714_send_scpi (instrID, func_input_ref_point[ref_point]) != 0)
     return bp9714_err;

    if (bp9714_send_scpi (instrID, func_input_filter[filter]) != 0)
     return bp9714_err;

    if (bp9714_send_scpi (instrID, func_input_range[range]) != 0)
     return bp9714_err;

    return (0);
}


/* ========================================================================= */
/*  This function reads the input function of the device                     */
/* ========================================================================= */
int _VI_FUNC bp9714_read_config_input(int instrID,int *ref_point,int *filter,int *range)
{
    char i;
    char hilf[10];
    bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;
   
    if(bp9714_receive_scpi(instrID,"INP:TYPE?",buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%s",hilf);     
           
	/* Decode reference point */

	if(strcmp(hilf,"BAL")==0)
		*ref_point = 1;		  		/* balanced */
	else 
		*ref_point = 0;		  		/* unbalanced */

	if(bp9714_receive_scpi(instrID,"INP:FILT?",buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%s",hilf);     
     
	if(strcmp(hilf,"OFF")==0)
		*filter = 0;		   		/* filter off */
	else 
		*filter = 1;		   		/* filter on */

	if(bp9714_receive_scpi(instrID,"SENS:VOLT:RANG?",buf) != 0)
      return bp9714_err;

	/* Decode input range */      
           
	if(strcmp(buf,"1.25 mV ")==0)
		*range= 0;
	else 
	if(strcmp(buf,"2.5 mV  ")==0)
		*range= 1;
	else
	if(strcmp(buf,"5 mV    ")==0)
		*range= 2;
	else
	if(strcmp(buf,"10 mV   ")==0)
		*range= 3;
	else
	if(strcmp(buf,"12.5 mV ")==0)
		*range= 4;
	else
	if(strcmp(buf,"25 mV   ")==0)
		*range= 5;
	else
	if(strcmp(buf,"50 mV   ")==0)
		*range= 6;
	else
	if(strcmp(buf,"100 mV  ")==0)
		*range= 7;
	else
	if(strcmp(buf,"125 mV  ")==0)
		*range= 8;
	else
	if(strcmp(buf,"250 mV  ")==0)
		*range= 9;
	else
	if(strcmp(buf,"500 mV  ")==0)
		*range= 10;
	else
	if(strcmp(buf,"1 V     ")==0)
		*range= 11;
	else
	if(strcmp(buf,"1.25 V  ")==0)
		*range= 12;
	else
	if(strcmp(buf,"2.5 V   ")==0)
		*range= 13;
	else
	if(strcmp(buf,"5 V     ")==0)
		*range= 14;
	else
	if(strcmp(buf,"10 V    ")==0)
		*range= 15;
	else
		*range = 16;		/* 12 V */

    return (0);
}


/* ========================================================================= */
/*  This function switches the comparator state ON/OFF                       */
/* ========================================================================= */
int _VI_FUNC bp9714_comp_state(int instrID,int state)
{
    bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;
    if (bp9714_invalid_integer_range (state, 0, 1,  -2) != 0)
     return bp9714_err;

    Fmt (cmd, "%s<CALC:LIM:STAT %i",state);
    if (bp9714_send_scpi (instrID, cmd) != 0)
     return bp9714_err;

    return (0);
}


/* ========================================================================= */
/*  This function reads the comparator state                                 */
/* ========================================================================= */
int _VI_FUNC bp9714_read_comp_state(int instrID,int *state)
{
    char hilf[10];
    bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;
 
	if(bp9714_receive_scpi(instrID,"CALC:LIM:STAT?",buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%d",state);     /* 0 = off, 1 = on */
           
    return (0);
}


/* ========================================================================= */
/*  This function reads the comparator state                                 */
/* ========================================================================= */
int _VI_FUNC bp9714_read_comp_count(int instrID,int *count)
{
    char hilf[10];
    bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;
 
	if(bp9714_receive_scpi(instrID,"CALC:LIM:COUNT?",buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%d",count);     /* 0 = off, 1 = on */
           
    return (0);
}


/* ========================================================================= */
/*  This function configures the comparator function of the device           */
/* ========================================================================= */
int _VI_FUNC bp9714_comp_config(int instrID,int comp_type,int comp_ctrl)
{
    bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;
    if (bp9714_invalid_integer_range (comp_type,0 , 1 , -2) != 0)
     return bp9714_err;
    if (bp9714_invalid_integer_range (comp_ctrl,0 , 3 , -3) != 0)
     return bp9714_err;

    if (bp9714_send_scpi (instrID, func_comp_type[comp_type]) != 0)
     return bp9714_err;

    if (bp9714_send_scpi (instrID, func_comp_ctrl[comp_ctrl]) != 0)
     return bp9714_err;

    return (0);
}


/* ========================================================================= */
/*  This function reads the comparator function of the device                */
/* ========================================================================= */
int _VI_FUNC bp9714_read_comp_config(int instrID,int *comp_type,int *comp_ctrl)
{
    char hilf[10];
    bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;
    
	if(bp9714_receive_scpi(instrID,"CALC:LIM:TYPE?",buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%s",hilf);     
           
	if(strcmp(hilf,"STAT")==0)
		*comp_type = 1;			   /* static */
	else 
		*comp_type = 0;			   /* dynamic */

	if(bp9714_receive_scpi(instrID,"CALC:LIM:CONT?",buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%s",hilf);     
           
	if(strcmp(hilf,"AVERAGE")==0)
		*comp_ctrl = 1;			   
	else 

	if(strcmp(hilf,"SINGLE")==0)
		*comp_ctrl = 0;
	else 
	
	if(strcmp(hilf,"MINIMUM")==0)
		*comp_ctrl = 2;
	else 

	if(strcmp(hilf,"MAXIMUM")==0)
		*comp_ctrl = 3;

    return (0);
}



/* ========================================================================= */
/*  This function configures the two limit comparator                        */
/* ========================================================================= */
int _VI_FUNC bp9714_comp_two_limit(int instrID,double lower,double upper)
{
    char cbuf[10];
    int flag;

    bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;

    Fmt (cmd, "%s<CALC:LIM:COUN 2"); /* two Limit Comparator*/
    if (bp9714_send_scpi (instrID, cmd) != 0)
     return bp9714_err;

    Fmt (cmd, "%s<CALC:LIM:LOW0 %f",lower);
    if (bp9714_send_scpi (instrID, cmd) != 0)
     return bp9714_err;

    Fmt (cmd, "%s<CALC:LIM:UPP0 %f",upper);
    if (bp9714_send_scpi (instrID, cmd) != 0)
     return bp9714_err;

    if(bp9714_receive_scpi(instrID,"CALC:LIM:ACKNOWLEDGE?",cbuf) != 0)/* Flag lesen */
     return bp9714_err;

    Scan(cbuf,"%s>%i",&flag);
    if(flag == 0)    						/* No Correct Comparator values*/
     return (330);

    return (0);
}


/* ========================================================================= */
/*  This function reads the two limit comparator                             */
/* ========================================================================= */
int _VI_FUNC bp9714_read_comp_two_limit(int instrID,double *lower,double *upper)
{
    char cbuf[10];
    int flag;

    bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;

	if(bp9714_receive_scpi(instrID,"CALC:LIM:LOW0?",buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%f",lower);     

	if(bp9714_receive_scpi(instrID,"CALC:LIM:UPP0?",buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%f",upper);     
       
    return (0);
}


/* ========================================================================= */
/*  This function configures the four limit comparator                       */
/* ========================================================================= */
int _VI_FUNC bp9714_comp_four_limit(int instrID,double lower1,double lower2,
                           double upper1,double upper2)
{
    char cbuf[10];
    int flag;

    bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;

    Fmt (cmd, "%s<CALC:LIM:COUN 4"); 		/* Four Limit Comparator*/
    if (bp9714_send_scpi (instrID, cmd) != 0)
     return bp9714_err;

    Fmt (cmd, "%s<CALC:LIM:LOW1 %f",lower1);
    if (bp9714_send_scpi (instrID, cmd) != 0)
     return bp9714_err;

    Fmt (cmd, "%s<CALC:LIM:UPP1 %f",upper1);
    if (bp9714_send_scpi (instrID, cmd) != 0)
     return bp9714_err;

    Fmt (cmd, "%s<CALC:LIM:LOW2 %f",lower2);
    if (bp9714_send_scpi (instrID, cmd) != 0)
     return bp9714_err;

    Fmt (cmd, "%s<CALC:LIM:UPP2 %f",upper2);
    if (bp9714_send_scpi (instrID, cmd) != 0)
     return bp9714_err;

    if(bp9714_receive_scpi(instrID,"CALC:LIM:ACKNOWLEDGE?",cbuf) != 0)/* read flag */
     return bp9714_err;

    Scan(cbuf,"%s>%i",&flag);
    if(flag == 0)    						/* No Correct Comparator values*/
     return (330);

    return (0);
}


/* ========================================================================= */
/*  This function reads the four limit comparator                            */
/* ========================================================================= */
int _VI_FUNC bp9714_read_comp_four_limit(int instrID,double *lower1,double *lower2,
                                double *upper1,double *upper2)
{
    char cbuf[10];
    int flag;

    bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;


	if(bp9714_receive_scpi(instrID,"CALC:LIM:LOW1?",buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%f",lower1);     
     

	if(bp9714_receive_scpi(instrID,"CALC:LIM:UPP1?",buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%f",upper1);     
     
	if(bp9714_receive_scpi(instrID,"CALC:LIM:LOW2?",buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%f",lower2);     

   	if(bp9714_receive_scpi(instrID,"CALC:LIM:UPP2?",buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%f",upper2);     
     
    return (0);
}


/* ========================================================================= */
/*  This function switches the peak detection state ON/OFF                   */
/* ========================================================================= */
int _VI_FUNC bp9714_peak_detection_state(int instrID,int state)
{
    bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;
    if (bp9714_invalid_integer_range (state, 0, 1,  -2) != 0)
     return bp9714_err;

    Fmt (cmd, "%s<SENS:EXTR:STAT %i",state);
    if (bp9714_send_scpi (instrID, cmd) != 0)
     return bp9714_err;

    return (0);
}


/* ========================================================================= */
/*  This function reads the peak detection state                             */
/* ========================================================================= */
int _VI_FUNC bp9714_read_peak_detection_state(int instrID,int *state)
{
    char hilf[10];
    bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;

	if(bp9714_receive_scpi(instrID,"SENS:EXTR:STAT?",buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%d",state);     
           
    return (0);
}


/* ========================================================================= */
/*  This function reads the peak values                                      */
/* ========================================================================= */
int _VI_FUNC bp9714_read_peak_values(int instrID,double* min_value,double* max_value,
                            double* diff_value)
{
    bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;

    if(bp9714_receive_scpi(instrID,"SENS:EXTR:DATA?",buf) != 0)
      return bp9714_err;

    Scan(buf,"%s>%f,%f,%f",min_value,max_value,diff_value);

    return (0);
}


/* ========================================================================= */
/*  This function switches the tare state ON/OFF                             */
/* ========================================================================= */
int _VI_FUNC bp9714_tare_state(int instrID,int state)
{
    bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;
    if (bp9714_invalid_integer_range (state, 0, 1,  -2) != 0)
     return bp9714_err;

    Fmt (cmd, "%s<SENS:CORR:OFFS %i",state);
    if (bp9714_send_scpi (instrID, cmd) != 0)
     return bp9714_err;

    return (0);
}


/* ========================================================================= */
/*  This function reads the tare state                                       */
/* ========================================================================= */
int _VI_FUNC bp9714_read_tare_state(int instrID,int *state)
{
    bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;

  	if(bp9714_receive_scpi(instrID,"SENS:CORR:OFFS?",buf) != 0)
      return bp9714_err;

    Scan(buf,"%s>%d",state);  
    
    return (0);
}


/* ========================================================================= */
/*  This function read the tare value                                        */
/* ========================================================================= */
int _VI_FUNC bp9714_read_tare_value(int instrID,double* value)
{
    bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;

   if(bp9714_get_tare_state(instrID) != 0)/* Warten auf Tarierung fertig*/
    return bp9714_err;

    if(bp9714_receive_scpi(instrID,"SENS:CORR:DATA?",buf) != 0)
      return bp9714_err;

    Scan(buf,"%s>%f",value);

    return (0);
}


/* ========================================================================= */
/*  This function starts the measure                                         */
/* ========================================================================= */
int _VI_FUNC bp9714_start_measure(int instrID)
{
    double t1,t2,diff;
    int ret;

    bp9714_err = 0;
	diff = 0.0; 

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;

    Fmt (cmd, "%s<INIT");
    if (bp9714_send_scpi (instrID, cmd) != 0)
      return bp9714_err;

    ret = -3;
    t1= Timer();                                
    while((ret != 0) && (diff < 15.0F)) 
    {
     Delay(0.1F);
     ret = bp9714_get_status(instrID);         
     t2= Timer();
     diff = fabs(t2-t1);
    }
    if(diff >= 15.0F)
     return (-3);

    return (ret);
}


/* ========================================================================= */
/*  This function reads the measurement value                                */
/* ========================================================================= */
int _VI_FUNC bp9714_read_measure(int instrID,double *value,char* unit,char * comp_value )
{
    double wert;
    char mw_buf[50],einheit[50],h_buf[30],dummy[20];
    int l, ret;
    double t1,t2,diff;   
    
    bp9714_err = 0;
	diff = 0.0; 

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;

    ret = 301;
    t1= Timer();                                
    bp9714_err = 0;

    while((ret==301) && (diff < 30.0F))
    {
	   	ret = (bp9714_receive_scpi(instrID,"fetch?",mw_buf) );
     	t2= Timer();
     	diff = fabs(t2-t1);
    }
 																			/* Changed on June 21. 2000 - ST */
/* 	if((diff >= 10.0F)||(ret!= 0))
    {
       bp9714_err = 351;    
	   return bp9714_err;
	}
*/
 	if(diff >= 10.0F)														/* Changes : */														
    {
       bp9714_err = 351;    
	   return bp9714_err;
	}
 	
 	if(ret!= 0)
	   return ret;															/* ------------------------------ */

    Scan(mw_buf,"%s>%s[t44],%s",h_buf,comp_value);
    Scan(h_buf,"%s>%f%s",value,unit);

    return (0);
}


/* ========================================================================= */
/*  This function reads the measurement value                                */
/* ========================================================================= */
int _VI_FUNC bp9714_read_string_measure(int instrID,char *value,char * comp_value )
{
    double wert;
    char mw_buf[50],einheit[50],h_buf[30],dummy[20];
    int l, ret;
    double t1,t2,diff;   
   
    bp9714_err = 0;
	diff = 0.0; 

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;

    ret = 301;
    t1= Timer();                                
    bp9714_err = 0;

    while((ret==301) && (diff < 30.0F))
    {
	   	ret = (bp9714_receive_scpi(instrID,"fetch?",mw_buf) );
     	t2= Timer();
     	diff = fabs(t2-t1);
    }
 																			/* Changed on June 21. 2000 - ST */
/* 	if((diff >= 10.0F)||(ret!= 0))
    {
       bp9714_err = 351;    
	   return bp9714_err;
	}
*/
 	if(diff >= 10.0F)														/* Changes : */														
    {
       bp9714_err = 351;    
	   return bp9714_err;
	}
 	
 	if(ret!= 0)
	   return ret;															/* ------------------------------ */

    Scan(mw_buf,"%s>%s[t44],%s",h_buf,comp_value);
    Scan(h_buf,"%s>%s",value);


    return (0);
}


/* ========================================================================= */
/*  This function stops the measure                                          */
/* ========================================================================= */
int _VI_FUNC bp9714_stop_measure(int instrID)
{
    bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;

    Fmt (cmd, "%s<ABORT");
    
    if (bp9714_send_scpi (instrID, cmd) != 0)
      return bp9714_err;

    return (0);
}


/* ========================================================================= */
/*  This function sets the Serial Communication Parameters                   */
/* ========================================================================= */
int _VI_FUNC bp9714_system_serial_set_para(int instrID,int i_type,int g_adr,int u_adr,
                                  int baudrate,int d_bits,int parity_mode,
                                  int bcc_mode,int c_delay_mode)
{
    bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;
    if (bp9714_invalid_integer_range (i_type, 0, 2,  -2) != 0)
     return bp9714_err;
    if (bp9714_invalid_integer_range (g_adr, 0, 15,  -3) != 0)
     return bp9714_err;
    if (bp9714_invalid_integer_range (u_adr, 0, 15,  -4) != 0)
     return bp9714_err;
    if (bp9714_invalid_integer_range (baudrate, 0, 7,  -5) != 0)
      return bp9714_err;
    if (bp9714_invalid_integer_range (d_bits, 0, 4,  -6) != 0)
     return bp9714_err;
    if (bp9714_invalid_integer_range (parity_mode, 0, 1,  -7) != 0)
     return bp9714_err;
    if (bp9714_invalid_integer_range (bcc_mode, 0, 1,  -8) != 0)
     return bp9714_err;
    if (bp9714_invalid_integer_range (c_delay_mode, 0, 1,  -9) != 0)
     return bp9714_err;

    Fmt (cmd, "%s<SYST:COMM:SER:TYPE %i",i_type);
    if (bp9714_send_scpi (instrID, cmd) != 0)
     return bp9714_err;

    Fmt (cmd, "%s<SYST:COMM:SER:ADDR:GRO %i",g_adr);
    if (bp9714_send_scpi (instrID, cmd) != 0)
     return bp9714_err;

    Fmt (cmd, "%s<SYST:COMM:SER:ADDR:USER %i",u_adr);
    if (bp9714_send_scpi (instrID, cmd) != 0)
     return bp9714_err;

    Fmt (cmd, "%s<SYST:COMM:SER:TRAN:BAUD %i",baudrate);
    if (bp9714_send_scpi (instrID, cmd) != 0)
     return bp9714_err;

    Fmt (cmd, "%s<SYST:COMM:SER:TRAN:BITS %i",d_bits);
    if (bp9714_send_scpi (instrID, cmd) != 0)
     return bp9714_err;

    Fmt (cmd, "%s<SYST:COMM:SER:TRAN:PAR %i",parity_mode);
    if (bp9714_send_scpi (instrID, cmd) != 0)
     return bp9714_err;
    
    Fmt (cmd, "%s<SYST:COMM:SER:BCC %i",bcc_mode);
    if (bp9714_send_scpi (instrID, cmd) != 0)
     return bp9714_err;

    Fmt (cmd, "%s<SYST:COMM:SER:CDEL %i",c_delay_mode);
    if (bp9714_send_scpi (instrID, cmd) != 0)
     return bp9714_err;

    return 0;
}


/* ========================================================================= */
/*  This function reads the Serial Communication Parameters                  */
/* ========================================================================= */
int _VI_FUNC bp9714_read_system_serial_para(int instrID,int *i_type,int *g_adr,int *u_adr,
                                  int *baudrate,int *d_bits,int *parity_mode,
                                  int *bcc_mode,int *c_delay_mode)
{
    bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;
  

   	if(bp9714_receive_scpi(instrID,"SYST:COMM:SER:TYPE?",buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%d",i_type);     	/* Interface type */
   

	if(bp9714_receive_scpi(instrID,"SYST:COMM:SER:ADDR:GRO?",buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%d",g_adr);      	/* group address */
   

	if(bp9714_receive_scpi(instrID,"SYST:COMM:SER:ADDR:USER?",buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%d",u_adr);       	/* user address */
   

	if(bp9714_receive_scpi(instrID,"SYST:COMM:SER:TRAN:BAUD?",buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%d",baudrate);     /* baudrate */
   

	if(bp9714_receive_scpi(instrID,"SYST:COMM:SER:TRAN:BITS?",buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%d",d_bits);       /* data bits */
   

	if(bp9714_receive_scpi(instrID,"SYST:COMM:SER:TRAN:PAR?",buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%d",parity_mode);   /* parity */  
   
   
   	if(bp9714_receive_scpi(instrID,"SYST:COMM:SER:BCC?",buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%d",bcc_mode);      /* blockcheck */
   

	if(bp9714_receive_scpi(instrID,"SYST:COMM:SER:CDEL?",buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%d",c_delay_mode);   /* character delay */  
   

    return 0;
}


/* ========================================================================= */
/*  This function accepts the  serial datas                                  */
/* ========================================================================= */
int _VI_FUNC bp9714_system_serial_accept(int instrID)
{
    bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;

    if(bp9714_send_scpi(instrID,"SYST:COMM:SER:ALL") != 0)/* Accept Datas */
     return bp9714_err;

    return (0);
}


/* ========================================================================= */
/*  This function stores memory no                                           */
/* ========================================================================= */
int _VI_FUNC bp9714_store_memory(int instrID,int mem_no)
{
    int ret;

    bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;
    if (bp9714_invalid_integer_range (mem_no, 1, 16,  -2) != 0)
     return bp9714_err;

    Fmt (cmd, "%s<*SAV %i",mem_no);
    if (bp9714_send_scpi (instrID, cmd) != 0)
      return bp9714_err;

    return (0);
}


/* ========================================================================= */
/*  This function recalls memory no                                          */
/* ========================================================================= */
int _VI_FUNC bp9714_recall_memory(int instrID,int mem_no)
{
    int ret;

    bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;
    if (bp9714_invalid_integer_range (mem_no, 0, 17,  -2) != 0)
     return bp9714_err;

    Fmt (cmd, "%s<*RCL %i",mem_no);
    
    if (bp9714_send_scpi (instrID, cmd) != 0)
      return bp9714_err;

    return (0);
}


/* ========================================================================= */
/*  This function sets the display contrast                                  */
/* ========================================================================= */
int _VI_FUNC bp9714_set_display_contrast(int instrID,double value)
{
    bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
    return bp9714_err;
    if (bp9714_invalid_real_range (value, 0.0F, 1.0F,  -2) != 0)
     return bp9714_err;

    Fmt (cmd, "%s<DISP:CONT %f",value);
    
    if (bp9714_send_scpi (instrID, cmd) != 0)
      return bp9714_err;

    return 0;
}


/* ========================================================================= */
/*  This function reads the display contrast                                 */
/* ========================================================================= */
int _VI_FUNC bp9714_read_display_contrast(int instrID,double *value)
{
    bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
    return bp9714_err;

    if(bp9714_receive_scpi(instrID,"DISP:CONT?",buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%f",value);     
   
    return 0;
}


/* ========================================================================= */
/*  This function returns the System Error                                   */
/* ========================================================================= */
int _VI_FUNC bp9714_system_error(int instrID,char* system_error)
{
    bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
    return bp9714_err;

    if(bp9714_receive_scpi(instrID,"SYST:ERR?",system_error) != 0)
     return bp9714_err;
    return 0;
}


/* ========================================================================= */
/*  This function returns the System Error                                   */
/* ========================================================================= */
int _VI_FUNC bp9714_system_keyboard(int instrID,int flag)
{
    bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
    return bp9714_err;
    if (bp9714_invalid_integer_range (flag, 0, 1,  -2) != 0)
    return bp9714_err;

    if (bp9714_send_scpi (instrID, func_system_keyboard[flag]) != 0)
     return bp9714_err;

    return 0;
}


/* ========================================================================= */
/*  This function sends the security code                                    */
/* ========================================================================= */
int _VI_FUNC bp9714_system_security(int instrID,int code)
{
    bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;
    if (bp9714_invalid_integer_range (code, 0, 9999,  -2) != 0)
     return bp9714_err;

    Fmt (cmd, "%s<SYST:SEC %i",code);
    if (bp9714_send_scpi (instrID, cmd) != 0)
     return bp9714_err;

    return 0;
}


/* ========================================================================= */
/*  This function reads the security code                                    */
/* ========================================================================= */
int _VI_FUNC bp9714_read_system_security(int instrID,int *code)
{
    bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;
  
    if(bp9714_receive_scpi(instrID,"SYST:SEC?",buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%d",code);     
   
    return 0;
}


/*=========================================================================*/
/* This function checks the device dependent error flag                    */
/*=========================================================================*/
int bp9714_get_eoc(int instrID)
{
    int ret,meas_status;
    double t1,t2,diff;

    ret = 0;
    t1= Timer();                                
    bp9714_err = 0;
	diff = 0.0; 

    while((ret != 1) && (diff < 30.0F))
    {
  
     if(bp9714_receive_scpi(instrID,":STAT:OPER:COND?",buf) != 0)
      return bp9714_err;
     Scan(buf,"%s>%i",&meas_status);

     if(meas_status & bp9714_DEVICE_EOC)
     {
      ret = 1;
      bp9714_err = 0;
     }

     t2= Timer();
     diff = fabs(t2-t1);
     Delay(0.1F); 
     
    }
 	if(diff >= 30.0F)
       bp9714_err = 351;    

    return (bp9714_err);
}


/*=========================================================================*/
/* This function checks the tare state                                     */
/*=========================================================================*/
int bp9714_get_tare_state(int instrID)
{
    int ret,tare_status;
    double t1,t2,diff;

    ret = 0;
    t1= Timer();                                
    bp9714_err = 0;
	diff = 0.0; 

    while((ret != 1) && (diff < 30.0F))
    {
  
     if(bp9714_receive_scpi(instrID,":STAT:OPER:COND?",buf) != 0)
      return bp9714_err;
     Scan(buf,"%s>%i",&tare_status);

     if((tare_status & bp9714_DEVICE_TARE))
     {
      ret = 1;
      bp9714_err = 0;
     }

     t2= Timer();
     diff = fabs(t2-t1);
     Delay(0.1F); 
     
    }
 	if(diff >= 30.0F)
       bp9714_err = 351;    

    return (bp9714_err);
}


/*=========================================================================*/
/* This function checks the error state                                    */
/*=========================================================================*/
int _VI_FUNC bp9714_get_status(int instrID)
{
      int status_flag,quest_flag;

      bp9714_err = 0;

      if(bp9714_receive_scpi(instrID,"*ESR?",buf) != 0)
       return bp9714_err;

      Scan(buf,"%s>%i",&status_flag);
      if(status_flag & bp9714_DEVICE_DEPT_ERROR)
      {
       if(status_flag & bp9714_EXECUTION_ERROR)
       {
        bp9714_err = 309;    	/* Execution ERROR */
        return bp9714_err;
       }

       if(bp9714_receive_scpi(instrID,"STAT:QUES:COND?",buf) != 0)
        return bp9714_err;

       Scan(buf,"%s>%i",&quest_flag);

       if(quest_flag == 0)
       {
        bp9714_err = 0;    		/* NO ERROR */
        return bp9714_err;
       }
       if(quest_flag & bp9714_VOLTAGE_ERROR)
       {
        bp9714_err = 310;    	/* Voltage ERROR */
        return bp9714_err;
       }
       if(quest_flag & bp9714_CURRENT_ERROR)
       {
        bp9714_err = 311;    	/* Current ERROR */
        return bp9714_err;
       }
      }
      return bp9714_err;
}


/*=========================================================================*/
/* This function checks the operation status register                      */
/*=========================================================================*/
int _VI_FUNC bp9714_get_cal_status(int instrID,int* status)
{
      int cal_status;

      bp9714_err = 0;

      if(bp9714_receive_scpi(instrID,"STAT:OPER?",buf) != 0)
       return bp9714_err;

      Scan(buf,"%s>%i",&cal_status);
      if(cal_status & bp9714_DEVICE_CALIBRATION)
       *status = 1;
      else
       *status = 0;

      return bp9714_err;
}


/*=========================================================================*/
/* This function checks the cal error registers                            */
/*=========================================================================*/
int _VI_FUNC bp9714_get_cal_err(int instrID,int* err_status,int* err_code)
{
      int status;

      *err_status = 0;
      *err_code = 0;
      bp9714_err = 0;

      if(bp9714_receive_scpi(instrID,"CST:I?",buf) != 0)
       return bp9714_err;

      Scan(buf,"%s>%i",&status);
      if(status != 0)
      {
       *err_status = 400;
       *err_code = status;
       return bp9714_err;
      }

      if(bp9714_receive_scpi(instrID,"CST:U:IN?",buf) != 0)
       return bp9714_err;
      Scan(buf,"%s>%i",&status);
      if(status != 0)
      {
       *err_status = 401;
       *err_code = status;
       return bp9714_err;
      }

      if(bp9714_receive_scpi(instrID,"CST:U:OUT?",buf) != 0)
       return bp9714_err;
      Scan(buf,"%s>%i",&status);
      if(status != 0)
      {
       *err_status = 402;
       *err_code = status;
       return bp9714_err;
      }


      if(bp9714_receive_scpi(instrID,"CST:R:IN?",buf) != 0)
       return bp9714_err;
      Scan(buf,"%s>%i",&status);
      if(status != 0)
      {
       *err_status = 403;
       *err_code = status;
       return bp9714_err;
      }

      if(bp9714_receive_scpi(instrID,"CST:R:OUT?",buf) != 0)
       return bp9714_err;
      Scan(buf,"%s>%i",&status);
      if(status != 0)
      {
       *err_status = 404;
       *err_code = status;
       return bp9714_err;
      }
      return bp9714_err;
}


/*=========================================================================*/
/* This function switches the logg state on/off                            */
/*=========================================================================*/
int _VI_FUNC bp9714_logg_state (int instrID, int datenlogger)
{
     bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;
    
    Fmt (cmd, "%s<DAT:STAT %i",datenlogger);
    if (bp9714_send_scpi (instrID, cmd) != 0)
     return bp9714_err;

    return (0);

 }


/*=========================================================================*/
/* This function reads the logg state                                      */
/*=========================================================================*/
int _VI_FUNC bp9714_read_logg_state (int instrID, int *dataloggerstate)
{
    bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;
   
    if(bp9714_receive_scpi(instrID,"DAT:STAT?",buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%d",dataloggerstate); 

    return (0);
 
}


/*=========================================================================*/
/* This function clears a block                                            */
/*=========================================================================*/
int _VI_FUNC bp9714_clear_block (int instrID, int block)
{
     bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;
   
    Fmt (cmd, "%s<DAT:CLEAR %i",block);
    if (bp9714_send_scpi (instrID, cmd) != 0)
     return bp9714_err;

    return (0);

 }


/*=========================================================================*/
/* This function selects a block                                           */
/*=========================================================================*/
int _VI_FUNC bp9714_select_block (int instrID, int block)
{
     bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;

    Fmt (cmd, "%s<DAT:SEL:BLOCK %i",block);
    if (bp9714_send_scpi (instrID, cmd) != 0)
     return bp9714_err;

    return (0);
}


/*=========================================================================*/
/* This function reads the selected block number                           */
/*=========================================================================*/
int _VI_FUNC bp9714_read_selected_block (int instrID, int *selectedBlock)
{
    bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;
   
    if(bp9714_receive_scpi(instrID,"DAT:SEL:BLOCK?",buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%d",selectedBlock); 

    return (0);
}


/*=========================================================================*/
/* This function reads the number of values in a block                     */
/*=========================================================================*/
int _VI_FUNC bp9714_read_count (int instrID, int *count, int block)
{
	unsigned char str[20];
    bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;
    sprintf(str,"%s %u","DAT:COUNT? ",block);
    if(bp9714_receive_scpi(instrID,str,buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%d",count); 

    return (0);
}


/*=========================================================================*/
/* This function reads the values of a block                               */
/*=========================================================================*/
int _VI_FUNC bp9714_read_data (int instrID, int block, int number,float *value)
{
	unsigned char str[40],unit[10];
    bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;
    sprintf(str,"%s %u,%u","DAT:DATA:ALL? ",block,number);
   
    if(bp9714_receive_scpi(instrID,str,buf) != 0)
      return bp9714_err;
  
	sscanf(buf,"%f%s",value,unit); 

    return (0);
}


/*=========================================================================*/
/* This function sets the parameters of a block                            */
/*=========================================================================*/
int _VI_FUNC bp9714_set_block_parameter (int instrID, int block, int size, char name[],
                                int filter, int XVal, double time, double delta)
{
     bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;
    
    Fmt (cmd, "%s<DAT:FILT:DELTA %i,%f",block,delta);
    if (bp9714_send_scpi (instrID, cmd) != 0)
     return bp9714_err;

    Fmt (cmd, "%s<DAT:SIZE %i,%i",block,size);
    if (bp9714_send_scpi (instrID, cmd) != 0)
     return bp9714_err;

    Fmt (cmd, "%s<DAT:STATE:DEF %s,%i",name,block);
    if (bp9714_send_scpi (instrID, cmd) != 0)
     return bp9714_err;

    Fmt (cmd, "%s<DAT:FILT:XVAL %i,%i",block,XVal);
    if (bp9714_send_scpi (instrID, cmd) != 0)
     return bp9714_err;

    Fmt (cmd, "%s<DAT:FILT:YTIME %i,%f",block,time);
    if (bp9714_send_scpi (instrID, cmd) != 0)
     return bp9714_err;

	switch(filter)
	{
	case 0:
	    Fmt (cmd, "%s<DAT:FILT %i,%s",block,"ALL");
	break;
	case 1:
	    Fmt (cmd, "%s<DAT:FILT %i,%s",block,"NOF");
	break;
	case 2:
	    Fmt (cmd, "%s<DAT:FILT %i,%s",block,"FAIL");
	break;
	case 3:
	    Fmt (cmd, "%s<DAT:FILT %i,%s",block,"XVAL");
	break;
	case 4:
	    Fmt (cmd, "%s<DAT:FILT %i,%s",block,"YTIME");
	break;
	case 5:
	    Fmt (cmd, "%s<DAT:FILT %i,%s",block,"DELTA");
	break;
	}
    if (bp9714_send_scpi (instrID, cmd) != 0)
     return bp9714_err;

	return (0);
}



/*=========================================================================*/
/* This function reads the parameters of a block                           */
/*=========================================================================*/
int _VI_FUNC bp9714_read_block_parameter (int instrID, int *size, int block,char name[], 
								 int *filter, int *XVal, double *time,double *delta)
{
	unsigned char str[20];
	unsigned char hilf[20];
     bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;

    sprintf(str,"%s %u","DAT:SIZE? ",block);
    if(bp9714_receive_scpi(instrID,str,buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%d",size); 

    sprintf(str,"%s %u","DAT:STATE:NAME? ",block);
    if(bp9714_receive_scpi(instrID,str,buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%s[t59]",name); 

    sprintf(str,"%s %u","DAT:FILT:XVAL? ",block);
    if(bp9714_receive_scpi(instrID,str,buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%d",XVal); 

    sprintf(str,"%s %u","DAT:FILT:YTIME? ",block);
    if(bp9714_receive_scpi(instrID,str,buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%f",time); 

    sprintf(str,"%s %u","DAT:FILT:DELTA? ",block);
    if(bp9714_receive_scpi(instrID,str,buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%f",delta); 

    sprintf(str,"%s %u","DAT:FILT? ",block);
	if(bp9714_receive_scpi(instrID,str,buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%s",hilf);  

	
	if((strcmp(hilf,"ALL")==0))
		*filter = 0;
	else
	if((strcmp(hilf,"NOF")==0))
		*filter = 1;
	else
	if((strcmp(hilf,"FAIL")==0))
		*filter = 2;
	else
	if((strcmp(hilf,"XVAL")==0))
		*filter = 3;
	else
	if((strcmp(hilf,"YTIM")==0))
		*filter = 4;
	else
	if((strcmp(hilf,"DELT")==0))
		*filter = 5;
	else
		*filter = 0;

  return(0);
}

/*=========================================================================*/
/* This function reads the evaluation data of a block                      */
/*=========================================================================*/
int _VI_FUNC bp9714_read_eval (int instrID,int block, double *minimum, double *stdDeviation,
                       double *maximum, double *meanValue)
{
	unsigned char str[20];
    bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;

    sprintf(str,"%s %u","DAT:MIN? ",block);
	if(bp9714_receive_scpi(instrID,str,buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%f",minimum);  

    sprintf(str,"%s %u","DAT:MAX? ",block);
	if(bp9714_receive_scpi(instrID,str,buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%f",maximum);  

    sprintf(str,"%s %u","DAT:AVER? ",block);
	if(bp9714_receive_scpi(instrID,str,buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%f",meanValue);  

    sprintf(str,"%s %u","DAT:DEV? ",block);
	if(bp9714_receive_scpi(instrID,str,buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%f",stdDeviation);  
	return(0);
}


/*=========================================================================*/
/* This function reads a block of data                                     */
/*=========================================================================*/ 
 int _VI_FUNC bp9714_read_data_block (int instrID,int block, int adresse,float data[]) 
 {
 	 unsigned char str[20];
     bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return bp9714_err;

    sprintf(str,"%s %u,%u","DA? ",block,adresse);
	if(bp9714_receive_scpi(instrID,str,buf) != 0)
      return bp9714_err;
  
	Scan(buf,"%s>%20f[xb4]",data);
	return(0);
}


/*=========================================================================*/
/* This function sends a SCPI- Message                                     */
/*=========================================================================*/
int _VI_FUNC bp9714_send_scpi(int instrID,char* send_buffer)
{
   bp9714_err = 0;

   Fmt(cmd,"%s<%s\n",send_buffer);

   bp9714_err = bp9714_send_x3_28(instrID,"sr",cmd);
   if(bp9714_err != 0)
    return (bp9714_err);

   return(0);
}
/*=========================================================================*/
/* This function receives a SCPI - Message                                 */
/*=========================================================================*/
int _VI_FUNC bp9714_receive_scpi(int instrID,char* send_buffer,char* receive_buffer)
{
   bp9714_err = 0;

   Fmt(cmd,"%s<%s\n",send_buffer);

   bp9714_err = bp9714_send_x3_28(instrID,"sr",cmd);
   if(bp9714_err != 0)
    return (bp9714_err);

   bp9714_err = bp9714_receive_x3_28(instrID,"po","",receive_buffer);
   if(bp9714_err != 0)
    return (bp9714_err);

   return(0);
}


/* ========================================================================= */
/*  This function closes the instrument.                                     */
/* ========================================================================= */
int _VI_FUNC bp9714_close (int instrID)
{
    bp9714_err = 0;

    if (bp9714_invalid_integer_range (instrID, 1, bp9714_MAX_INSTR,  -1) != 0)
     return (bp9714_err);

    if (bp9714_device_closed(instrID) != 0)
     return (bp9714_err);
    bp9714_close_instr(instrID);

    return (bp9714_err);
}


/* ===================== SERIAL UTILITY ROUTINES ========================== */


/* ========================================================================= */
/*  This function sends a string to the device                               */
/* ========================================================================= */
int com_wrt(int port,char * send_string)
{
      int ret,l;
      /* for CVI Only*/
	  if(ComType != bp9714_RS232)										/* If not RS232 */
         ComSetEscape(port,escape_on_bit);

      l = StringLength(send_string);
      ret = ComWrt(port, send_string, l);     /* Write the Message */

       /* for CVI Only*/
      while(GetOutQLen(port) != 0);
      Delay(escape_port_delay);  
	  if(ComType != bp9714_RS232)										/* If not RS232 */
         ComSetEscape(port,escape_off_bit);

      return ret;
}


/* ========================================================================= */
/*  This function sends one Byte to the device                               */
/* ========================================================================= */
int com_wrt_byte(int port,int send_byte)
{
      int ret;
      
      /* for CVI Only*/
	  if(ComType != bp9714_RS232)										/* If not RS232 */
         ComSetEscape(port,escape_on_bit);
	  
      ret =  ComWrtByte(port,send_byte);
      
       /* for CVI Only*/
      while(GetOutQLen(port) != 0);
      Delay(escape_port_delay);
	  if(ComType != bp9714_RS232)										/* If not RS232 */
         ComSetEscape(port,escape_off_bit);
      

      return ret;
}


/* ========================================================================= */
/*  This function locates and initializes an entry in the Instrument Table   */
/*  and the device table for the instrument.  If successful, the             */
/*  instrument ID is returned, else a -1 is returned.  The size of the       */
/*  Instrument Table can be changed in the include file by altering the      */
/*  constant bp9714_MAX_INSTR.                                               */
/* ========================================================================= */
int bp9714_open_instr(int port,long baud_rate,int g_addr,int u_addr)
{
    int i, instrID;

    instrID = 0;

    /*--Check to see if the instrument is already in the Instrument Table.-*/

    for (i=1; i< bp9714_MAX_INSTR; i++)
    {
     if((dev_group_addr[i] == g_addr) && (dev_user_addr[i] == u_addr))
     {
        instrID = i;
    i = bp9714_MAX_INSTR;
     }
    }

   /*------ If it is not in the instrument table, open an entry for the
            instrument. ---------------------------------------------------*/

    if (instrID <= 0)
    {
     for (i=1; i< bp9714_MAX_INSTR; i++)
     {
        if( (dev_group_addr[i] < 0) && (dev_user_addr[i] < 0))
        {
         instrID = i;
     i = bp9714_MAX_INSTR;
        }
     }
    }

    /*----- If an entry could not be opened in the Instrument Table,
            return an error.-----------------------------------------------*/

    if (instrID <= 0)
    {
     bp9714_err = 220;
     return -1;
    }

    dev_port[instrID] = port;

    /* CVI Function  */
    OpenComConfig (dev_port[instrID], "", baud_rate,0, 8, 1, 512, 512);

    /* DOS Function
    OpenComConfig (dev_port[instrID], baud_rate,0, 8, 1, 512, 512,
                                      port_addr[port-1],int_level[port-1]);*/
    if (rs232err != 0)
    {
     dev_port[instrID] = 0;
     dev_group_addr[instrID] = -1;
     dev_user_addr[instrID] = -1;
     bp9714_err = rs232err;
     return (-1);
    }
    dev_group_addr[instrID] = g_addr;
    dev_user_addr[instrID] = u_addr;
    return instrID;
}


/*=========================================================================*/
/* Function: Close Instrument                                              */
/* Purpose:  This function closes the instrument by removing it from the   */
/*           device table and setting the address and the bd to zero       */
/*           in the Instrument Table.  The return value is equal to the    */
/*           global error variable.                                        */
/*=========================================================================*/
int bp9714_close_instr(int instrID)
{
    if( (dev_group_addr[instrID] >= 0) && (dev_user_addr[instrID] >= 0))
    {
      dev_port[instrID] = 0;
      dev_group_addr[instrID] = -1;
      dev_user_addr[instrID] = -1;
    }
    else
     bp9714_err = 221;

    return bp9714_err;
 }


/*=========================================================================*/
/* Function: Device Closed                                                 */
/* Purpose:  This function checks to see if the module has been            */
/*           initialized.  If the device has not been opened, a 1 is       */
/*           returned, 0 otherwise.                                        */
/*=========================================================================*/
int bp9714_device_closed(int instrID)
{
     if( (dev_group_addr[instrID] < 0) && (dev_user_addr[instrID] < 0))
     {
    bp9714_err = 232;
        return -1;
     }
     return 0;
}



/*=========================================================================*/
/* This function sends a comand and message and waits for ACK              */
/*=========================================================================*/
int bp9714_send_x3_28(int instrID,char* cmd,char* msg)
{
    int ret,r_byte;

    bcc_repeat = 0;
    for(;;)
    {
     FlushInQ(dev_port[instrID]);   /* Clear Port */
     ret = bp9714_send_x3_28_message(instrID,cmd,msg);
     if(ret != 0)
       return (ret);     			/* ERROR*/

     r_byte = ComRdByte(dev_port[instrID]);
     if (rs232err != 0)
      return (231);

     switch(r_byte)
     {
      case ACK:
       return  (0);
//       break;
      case NAK:
       if(bcc_repeat >= max_bcc_repeat)
        return (300);				/* NAK => MAX Repeat */
       bcc_repeat++;
       break;
      case EOT:
       return (301);
//       break;
     }
    }

//    return (0);
}



/*=========================================================================*/
/* This function sends comand and message and receives a string            */
/*=========================================================================*/
int bp9714_receive_x3_28(int instrID,char* cmd,char* msg,char* receive_buf)
{
    int ret;

    FlushInQ(dev_port[instrID]);           	/* Clear Port */
    ret = bp9714_send_x3_28_message(instrID,cmd,msg);
    if(ret != 0)
      return (ret);     					/* ERROR*/

    ret = bp9714_receive_and_ack(instrID,receive_buf);
    if(ret != 0)
      return (ret);     					/* ERROR*/

    ComRdByte(dev_port[instrID]);   		/* Read the EOT from Port*/

    return(0);
}


/*=========================================================================*/
/* This function receives a string                                         */
/*=========================================================================*/
int bp9714_receive_and_ack(int instrID,char* receive_buf)
{
    int r_byte,r_bcc,l,first_byte;
	unsigned char bcc;
	
    bcc_repeat = 0;
    for(;;)
    {
     first_byte = ComRdByte(dev_port[instrID]);   /* Read the EOT or STX from Port*/
     if(first_byte == EOT)
      return (301);

     r_byte = ComRdTerm(dev_port[instrID],in_buf,bp9714_MAX_CMD-1,ETX);/* Read to ETX*/
     if (rs232err != 0)
      return (231);

     if(bcc_state == 1)
     {
      r_bcc = ComRdByte(dev_port[instrID]);   	/* Read the BCC from Port*/
      if (rs232err != 0)
       return (231);
     }
     in_buf[r_byte] = ETX;            			/* Calculate bcc with ETX */
     in_buf[r_byte+1] = NUL;          			/* Terminate the String*/
     l = StringLength(in_buf);

     CopyString (swap_buf, 0,in_buf, 0, l);   	/* calculate bcc without STX */
     if(bcc_state == 1)
     {
      bcc = bp9714_calculate_bcc(swap_buf);
      if(r_bcc != bcc)
      {
       com_wrt_byte(dev_port[instrID],NAK);
       if(bcc_repeat >= max_bcc_repeat)
        return (300);                			/* NAK => MAX Repeat */
       bcc_repeat++;
      }
      else

      {
       com_wrt_byte(dev_port[instrID],ACK);
       CopyString (receive_buf, 0,in_buf,0,l-3);/* Copy without LF,ETX*/
       return(0);                				/* 0 = ACK == OK*/
      }
     }
     else    									/* Without bcc check*/
     {
      com_wrt_byte(dev_port[instrID],ACK);
      CopyString (receive_buf, 0,in_buf,0,l-3); /* Copy without LF,ETX*/
      return(0);                				/* 0 = ACK == OK*/
     }
    }

//    return(0);
}


/*=========================================================================*/
/* This function sends a comand and message                                */
/*=========================================================================*/
int bp9714_send_x3_28_message(int instrID,char* cmd,char* msg)
{
    int ret;
    unsigned char bcc;

    ret = CompareStrings (msg, 0,"", 0, 0); 	/* Send a command without msg ?*/
    if(ret == 0)
     Fmt(out_buf,"%s<%c%x%x%x%x%s%c", EOT,
                                      dev_group_addr[instrID],
                                      dev_group_addr[instrID],
                                      dev_user_addr[instrID],
                                      dev_user_addr[instrID],
                                      cmd,ENQ); /* without msg*/
    else
    {
     Fmt(swap_buf ,"%s<%s%c",msg,ETX);   		/* with msg*/
     if(bcc_state == 1)
     {
      bcc = bp9714_calculate_bcc(swap_buf);
      Fmt(out_buf,"%s<%c%x%x%x%x%s%c%s%c", EOT,
                                           dev_group_addr[instrID],
                                           dev_group_addr[instrID],
                                           dev_user_addr[instrID],
                                           dev_user_addr[instrID],
                                           cmd,STX,swap_buf,bcc);
     }
     else
      Fmt(out_buf,"%s<%c%x%x%x%x%s%c%s", EOT,
                                         dev_group_addr[instrID],
                                         dev_group_addr[instrID],
                                         dev_user_addr[instrID],
                                         dev_user_addr[instrID],
                                         cmd,STX,swap_buf);
    }

    ret = com_wrt(dev_port[instrID], out_buf);

    if (rs232err != 0)
     return (231);

    return (0);
}


/*=========================================================================*/
/* This function calculates the block check                                */
/*=========================================================================*/
unsigned char bp9714_calculate_bcc(char* msg)
{
    int l,i;
	unsigned char bcc;
	
    bcc = 0;
    l = StringLength(msg);
    for(i = 0;i <= l;i++)
     bcc ^= msg[i];     		/* exclusive OR */

    bcc |= 0x80;    			/* OR Bit 8 */

    return (bcc);
}



/* ================ Global UTILITY ROUTINES ===============================*/

/*=========================================================================*/
/* Function: Invalid Integer Range                                         */
/* Purpose:  This function checks an integer to see if it lies between a   */
/*           minimum and maximum value.  If the value is out of range, set */
/*           the global error variable to the value err_code.  If the      */
/*           value is OK, error = 0.                                       */
/*=========================================================================*/
int bp9714_invalid_integer_range (int val,int min,int max,int err_code)
{
  if ((val < min) || (val > max))
  {
    bp9714_err = err_code;
    return -1;
  }
  return 0;
}


/*=========================================================================*/
/* Function: Invalid long Range                                            */
/* Purpose:  This function checks an long    to see if it lies between a   */
/*           minimum and maximum value.  If the value is out of range, set */
/*           the global error variable to the value err_code.  If the      */
/*           value is OK, error = 0.                                       */
/*=========================================================================*/
int bp9714_invalid_long_range (long val,long min,long max,int err_code)
{
  if ((val < min) || (val > max))
  {
    bp9714_err = err_code;
    return -1;
  }
  return 0;
}


/*=========================================================================*/
/* Function: Invalid Real Range                                            */
/* Purpose:  This function checks a real number to see if it lies between  */
/*           a minimum and maximum value.  If the value is out of range,   */
/*           set the global error variable to the value err_code.  If the  */
/*           value is OK, error = 0.                                       */
/*=========================================================================*/
int bp9714_invalid_real_range (double val,double min,double max,int err_code)
{
  if ((val < min) || (val > max))
  {
    bp9714_err = err_code;
    return -1;
  }
  return 0;
}




/*=========================================================================*/
/* Function: Setup the arrays                                              */
/* Purpose:  This function provides the arrays with default parameters.    */
/*=========================================================================*/
void bp9714_setup_arrays (void)
{
    int i;

    if(!setup_first)	/*----------- Reset the Arrays before initialize --------*/
    {
     for (i=0; i< bp9714_MAX_INSTR; i++)
     {
       dev_group_addr[i] = -1;
       dev_user_addr[i]  = -1;
       dev_port[i]       =  0;
     }
     setup_first = 1;
    }
    port_addr[0] = 0x3f8;
    port_addr[1] = 0x2f8;
    port_addr[2] = 0x3e8;
    port_addr[3] = 0x2e8;

    int_level[0] = 4;
    int_level[1] = 3;
    int_level[2] = 4;
    int_level[3] = 3;

    /*  SYSTEM Keyboard  */
    func_system_keyboard[0] = "SYST:KLOC OFF";
    func_system_keyboard[1] = "SYST:KLOC ON";

    /*  Calibration   */
    func_cal_input_meas[0] = "CAL:INIT:SENSE:LOW";
    func_cal_input_meas[1] = "CAL:INIT:SENSE:HIGH";

    func_cal_unit[0] = "CAL:MATH:EXPR 0";
    func_cal_unit[1] = "CAL:MATH:EXPR 1";

    func_aver_ctrl[0] = "SENS:AVER:TCON MOV";
    func_aver_ctrl[1] = "SENS:AVER:TCON REP";

    func_shape[0] = "SOUR:FUNC DC";
    func_shape[1] = "SOUR:FUNC LOW";
    func_shape[2] = "SOUR:FUNC DCR";
    func_shape[3] = "SOUR:FUNC ACR";
    func_shape[4] = "SOUR:FUNC ALTR";

    func_output_type[0] = "OUTP:TYPE UNB";
    func_output_type[1] = "OUTP:TYPE BAL";

    func_output_level[0] = "SOUR:VOLT 2.5V";
    func_output_level[1] = "SOUR:VOLT 5V";
    func_output_level[2] = "SOUR:VOLT 10V";
    func_output_level[3] = "SOUR:VOLT 5V";
    func_output_level[4] = "SOUR:VOLT 10V";
    func_output_level[5] = "SOUR:VOLT 20V";

    func_analog_type[0] = "SOUR:FUNC:MODE CURR";
    func_analog_type[1] = "SOUR:FUNC:MODE VOLT";

    func_input_ref_point[0] = "INP:TYPE UNB"; /* Grounded*/
    func_input_ref_point[1] = "INP:TYPE BAL"; /* Differential */

    func_input_filter[0] = "INP:FILT OFF";
    func_input_filter[1] = "INP:FILT ON";

    func_input_range[0] =  "SENS:VOLT:RANG 1.25MV";
    func_input_range[1] =  "SENS:VOLT:RANG 2.5MV";
    func_input_range[2] =  "SENS:VOLT:RANG 5MV";
    func_input_range[3] =  "SENS:VOLT:RANG 10MV";
    func_input_range[4] =  "SENS:VOLT:RANG 12.5MV";
    func_input_range[5] =  "SENS:VOLT:RANG 25MV";
    func_input_range[6] =  "SENS:VOLT:RANG 50MV";
    func_input_range[7] =  "SENS:VOLT:RANG 100MV";
    func_input_range[8] =  "SENS:VOLT:RANG 125MV";
    func_input_range[9] =  "SENS:VOLT:RANG 250MV";
    func_input_range[10] =  "SENS:VOLT:RANG 500MV";
    func_input_range[11] =  "SENS:VOLT:RANG 1V";
    func_input_range[12] =  "SENS:VOLT:RANG 1.25V";
    func_input_range[13] =  "SENS:VOLT:RANG 2.5V";
    func_input_range[14] =  "SENS:VOLT:RANG 5V";
    func_input_range[15] =  "SENS:VOLT:RANG 10V";
    func_input_range[16] =  "SENS:VOLT:RANG 12V";

    func_comp_type[0] = "CALC:LIM:TYPE DYN";
    func_comp_type[1] = "CALC:LIM:TYPE STAT";

    func_comp_ctrl[0] = "CALC:LIM:CONT SINGLE";
    func_comp_ctrl[1] = "CALC:LIM:CONT AVER";
    func_comp_ctrl[2] = "CALC:LIM:CONT MIN";
    func_comp_ctrl[3] = "CALC:LIM:CONT MAX";
}
